define(['BaseFHIRService'], function (BaseFHIRService) {
    'use strict';

    describe('Base FHIR Service', function () {
        var baseFHIRService,
            requestService,
            pageableObjectFactory,
            baseFHIRConfig = {
                getUrl: function () {
                    return 'http://localhost/fhir/'
                }
            };

        beforeEach(function () {
            requestService = jasmine.createSpyObj('requestService', ['sendRequest']);
            pageableObjectFactory = jasmine.createSpyObj('pageableObjectFactory', ['createPageableObject']);

            baseFHIRService = BaseFHIRService(requestService, baseFHIRConfig, pageableObjectFactory);
            baseFHIRService.type = 'Observation';
        });

        describe('getUrl function', function () {
            it('should return the default url if nothing is passed in', function () {
                var url = baseFHIRService.getUrl();

                expect(url).toEqual('http://localhost/fhir/Observation');
            });

            it('should return the url including the resource ID if provided', function () {
                var url = baseFHIRService.getUrl('someId');

                expect(url).toEqual('http://localhost/fhir/Observation/someId');
            });
        });

        describe('read function', function () {
            it('should throw an error if no ID is passed in', function () {
                try {
                    baseFHIRService.read();
                    fail('No error was thrown.');
                } catch (e) {
                    expect(e).toBeTruthy();
                    expect(e.message).toEqual('An ID must be supplied for a read request');
                }
            });

            it('should make a GET request', function () {
                baseFHIRService.read('someId');

                expect(requestService.sendRequest).toHaveBeenCalledWith({
                    method: 'GET',
                    url: 'http://localhost/fhir/Observation/someId'
                });
            });
        });

        describe('create function', function () {
            it('should throw an error if no resource is passed in', function () {
                try {
                    baseFHIRService.create();
                    fail('No error was thrown.');
                } catch (e) {
                    expect(e).toBeTruthy();
                    expect(e.message).toEqual('A resource object must be supplied for a create request');
                }
            });

            it('should throw an error if the resource is not an object', function () {
                try {
                    baseFHIRService.create('A string');
                    fail('No error was thrown.');
                } catch (e) {
                    expect(e).toBeTruthy();
                    expect(e.message).toEqual('A resource object must be supplied for a create request');
                }
            });

            it('should throw an error if the resource contains an id', function () {
                try {
                    baseFHIRService.create({ id: 'someId' });
                    fail('No error was thrown.');
                } catch (e) {
                    expect(e).toBeTruthy();
                    expect(e.message).toEqual('An ID element must not be supplied in the resource body on a create request');
                }
            });

            it('should make a POST request', function () {
                var resource = { resourceType: 'Observation' };
                
                baseFHIRService.create(resource);

                expect(requestService.sendRequest).toHaveBeenCalledWith({
                    method: 'POST',
                    url: 'http://localhost/fhir/Observation',
                    data: resource
                });
            });
        });

        describe('update function', function () {
            it('should throw an error if no resource is passed in', function () {
                try {
                    baseFHIRService.update();
                    fail('No error was thrown.');
                } catch (e) {
                    expect(e).toBeTruthy();
                    expect(e.message).toEqual('A resource object must be supplied for an update request');
                }
            });

            it('should throw an error if the resource is not an object', function () {
                try {
                    baseFHIRService.update('A string');
                    fail('No error was thrown.');
                } catch (e) {
                    expect(e).toBeTruthy();
                    expect(e.message).toEqual('A resource object must be supplied for an update request');
                }
            });

            it('should throw an error if the resource does not contain an id', function () {
                try {
                    baseFHIRService.update({ resourceType: 'Observation' });
                    fail('No error was thrown.');
                } catch (e) {
                    expect(e).toBeTruthy();
                    expect(e.message).toEqual('An ID element must be supplied in the resource body on an update request');
                }
            });

            it('should make a PUT request', function () {
                var resource = { resourceType: 'Observation', id: 'someId' };
                
                baseFHIRService.update(resource);

                expect(requestService.sendRequest).toHaveBeenCalledWith({
                    method: 'PUT',
                    url: 'http://localhost/fhir/Observation/someId',
                    data: resource
                });
            });
        });

        describe('delete function', function () {
            it('should throw an error if no ID is passed in', function () {
                try {
                    baseFHIRService.delete();
                    fail('No error was thrown.');
                } catch (e) {
                    expect(e).toBeTruthy();
                    expect(e.message).toEqual('An ID must be supplied for a delete request');
                }
            });

            it('should make a DELETE request', function () {
                baseFHIRService.delete('someId');

                expect(requestService.sendRequest).toHaveBeenCalledWith({
                    method: 'DELETE',
                    url: 'http://localhost/fhir/Observation/someId'
                });
            });
        });

        describe('search function', function () {
            var responseObj = {
                resourceType: 'Observation'
            };

            var PageableObject = function (params) {
                this.resourceType = params.resourceType;
            };

            beforeEach(function () {
                requestService.sendRequest.and.callFake(function () {
                    return {
                        then: function (callback) {
                            if (typeof callback === 'function') {
                                return callback(responseObj);
                            }
                        }
                    }
                });

                pageableObjectFactory.createPageableObject.and.callFake(function (response) {
                    return new PageableObject(response);
                });
            });

            it('should make a GET request with the given parameters', function () {
                baseFHIRService.search({_count: 50});

                expect(requestService.sendRequest).toHaveBeenCalledWith({
                    method: 'GET',
                    url: 'http://localhost/fhir/Observation',
                    params: {
                        _count: 50
                    }
                });
            });

            it('should create an instance of a PageableObject', function () {
                var pageableObject = baseFHIRService.search({_count: 50});

                expect(pageableObjectFactory.createPageableObject).toHaveBeenCalledWith(responseObj);
                expect(pageableObject instanceof PageableObject).toBeTruthy();
            });
        });
    });
});